#ifndef __CControl__
#define __CControl__

#include "IControl.hpp"
#include <Testing/CTrace.hpp>

//	===========================================================================

using Exponent::GUI::Controls::IControl;
using Exponent::Testing::CTrace;

//	===========================================================================

namespace Exponent
{
	namespace GUI
	{
		namespace Controls
		{
			/**
			 * @class CControl CControl.hpp
			 * @brief Basic implementation of a do nothing control
			 * 
			 * @date 19/03/2005
			 * @author Paul Chana
			 * @version 1.0.0 Initial version
			 * @version 1.0.1 Added drawPrimaryImage
			 * @version 1.0.2 Added NO_ID_REQUIRED static
			 *
			 * @note All contents of this source code are copyright 2005 Exp Digital Uk.\n
			 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy\n
			 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
			 * All content is the Intellectual property of Exp Digital Uk.\n
			 * Certain sections of this code may come from other sources. They are credited where applicable.\n
			 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
			 *
			 * $Id: CControl.hpp,v 1.9 2007/02/08 21:07:54 paul Exp $
			 */
			class CControl : public IControl
			{
				/** @cond */
				EXPONENT_CLASS_DECLARATION;
				/** @endcond */

//	===========================================================================
		
			public:

//	===========================================================================

				const static long CCONTROL_NO_ID_REQUIRED = -1;		/**< Control does not require unique id */

//	===========================================================================

				/**
				 * Construction
				 * @param root The root control that this control is being added to
				 * @param uniqueId The unique Identifier of this control or CCONTROL_NO_ID_REQUIRED if no id is required
				 * @param area The area of this control relative to the root controls top left
				 * @param listener The action listener
				 */
				CControl(IControlRoot *root, const long uniqueId, const CRect &area, IActionListener *listener = NULL);

				/**
				 * Destruction
				 */
				virtual ~CControl();

//	===========================================================================

				/**
				 * Set the root control that this control is sitting on
				 * @param controlRoot The control root that this control is a child of
				 */
				virtual void setControlRoot(IControlRoot *controlRoot);

				/**
				 * Get the root control
				 * @retval IControlRoot* The control root that this control is a child of
				 */
				virtual IControlRoot *getRootControl() const { return m_rootControl; }

				/**
				 * Set the parent control
				 * @param parent The parent control of this control, not necessarily == getControlRoot()
				 */
				virtual void setParentControl(IControl *parent) { m_parentControl = parent; }

				/**
				 * Get the parent control
				 * @retval IControl* The parent control of this control, not necessarily == getControlRoot()
				 */
				virtual IControl *getParentControl() const { return m_parentControl; }

//	===========================================================================

				/**
				 * Register the action listener
				 * @param listener The listener for events
				 */
				virtual void registerActionListener(IActionListener *listener) { m_actionListener = listener; }

//	===========================================================================

				/**
				 * Does the control accept mouse messages?
				 * @param mouseEnabled If true control accepts mouse events, if false doesnt
				 */
				virtual void controlIsMouseEnabled(const bool mouseEnabled = true) { m_mouseEnabled = mouseEnabled; }

				/**
				 * Does the control accept dropped files?
				 * @param dropEnabled If true control accepts drop file events, if false doesnt
				 */
				virtual void controlIsDropFileEnabled(const bool dropEnabled = true) { m_dropEnabled = dropEnabled; }

				/**
				 * Does the control accept key messages
				 * @param keyEnabled If true control accepts key events, if false doesnt
				 */
				virtual void controlIsKeyEnabled(const bool keyEnabled = true) { m_keyboardEnabled = keyEnabled; }

				/**
				 * Reserved for Future use
				 * @param identifier The future identifiers
				 */
				virtual void controlIsEnabledFor(const CString &identifier) { }

//	===========================================================================

				/**
				 * Is the control mouse enabled
				 * @retval bool True if the control accepts mouse events, false otherwise
				 */
				virtual bool isMouseEnabled() const { return m_mouseEnabled; }

				/**
				 * Is the control drop enabled
				 * @retval bool True if the control accepts drop file events, false otherwise
				 */
				virtual bool isDropFileEnabled() const { return m_dropEnabled; }

				/**
				 * Is the control key enabled
				 * @retval bool True if the control accepts key events, false otherwise
				 */
				virtual bool isKeyEnabled() const { return m_keyboardEnabled; }

				/**
				 * Is the control enabled for (future use)
				 * @param identifier The future identifiers
				 * @retval bool True if the control accepts the events, false otherwise
				 */
				virtual bool isEnabledFor(const CString &identifier) { return false; }

				/**
				 * Does the control require a right click feed?
				 * @retval bool True if the control require a right click, false otherwise
				 */
				virtual bool needsRightClick() const { return m_needsRightClick; }

//	===========================================================================

				/**
				 * Serialise the control from a filename
				 * @param node The node that you should setup
				 * @retval bool True if saved to node properly, false otherwise
				 * @note FOR FUTURE USE, DO NOT USE YET!
				 */
				virtual bool serializeToNode(CXMLNode &node);

				/**
				 * Serialise the control from a filename
				 * @param node The node that you should setup from
				 * @retval bool True if loaded from node properly, false otherwise
				 * @note FOR FUTURE USE, DO NOT USE YET!
				 */
				virtual bool serializeFromNode(CXMLNode &node);

//	===========================================================================

				/**
				 * Get the id of the control
				 * @retval long The unique id of the control
				 */
				virtual long getUniqueId() const { return m_uniqueId; }

				/**
				 * Set hte id of the control
				 * @param id The id of the control
				 */
				virtual void setUniqueId(const long id) { m_uniqueId = id; }

				/**
				 * Get the value of the control
				 * @retval double The value of the control ranged 0 - 1
				 */
				virtual double getValue() const { return m_value; }

				/**
				 * Set the vlaue of the control
				 * @param value The value of the control ranged 0 - 1
				 */
				virtual void setValue(const double value) { m_value = value; }

//	===========================================================================

				/**
				 * Set the area of the control
				 * @param area The area of the control relative to the root controls top left
				 */
				virtual void setArea(const CRect &area);

				/**
				 * Get the position relative to their root
				 * @retval const CRect& The position of the control relative to the root controls top left
				 */
				virtual const CRect &getArea() const { return m_area; }

				/**
				 * Get the position on the window
				 * @retval const CRect& The position of the control relative to the window hosting controls top left
				 */
				virtual const CRect &getAbsoluteRect() const { return m_absoluteArea; }

				/**
				 * Get the size of the control
				 * @retval const CDimension& The dimensions of the cotnrol
				 */
				virtual const CDimension &getDimension() const { return m_dimension; }

				/**
				 * Get the normalised control area (0, 0, width, height)
				 * @retval const CRect& The area of the control with a 0,0 top left
				 */
				virtual const CRect &getNormalisedArea() const { return m_normalisedArea; }

				/**
				 * Is this control a container of other controls 
				 * @retval bool True if this control is a container
				 * @see CControlPanel
				 * @see IControlRoot
				 */
				virtual bool isContainer() const { return false; }

//	===========================================================================

				/**
				 * Force Update the control (redraw it)
				 */
				virtual void update();

				/**
				 * Draw the control
				 * @param graphics The graphics context
				 * @note By the time a draw has been called on your control, the IControlRoot parent that holds your control will have setup\n
				 * the clipping area and updated the graphics context to be drawing relative to your top left. Note this again, because its very\n
				 * important <B>All drawing inside drawControl is done relative to your own top left</B>\n
				 * So for example if you wished to draw a box around your own entire area, you would use the following code:\n
				 * @code
				 * // Long hand drawing
				 * graphics.drawRectangle(CRect(0, 0, this->getArea().getWidth(), this->getArea().getHeight()));
				 * 
				 * // Note that for convenience, a normalised (0, 0 based) rect is setup for you during your construtor or calls to set rect.
				 * // This gives you a quick way to get this rectangle
				 * graphics.drawRectangle(m_normalisedArea);
				 * @endcode
				 * \n
				 * There are three key stages during the drawing of a control.\n
				 * <OL>
				 *		<LI>First you must check To see if the control must actually be drawn. You can do this in two ways. The default and normal way of checking for drawing is\n
				 *		    with a call to drawEnabledControl. If this returns true you are enabled to draw. This function will check if the control is enabled and if it is not, and there\n
				 *			is a disabled image (setDisabledImage) available, this will be drawn in to our area.\n
				 *			The second way that you can check if drawing should take place is to directly check the protected variable m_enabled. If this is true, you can draw normally. If this is false\n
				 *			you should draw your control in a way that indicates that it is disabled\n
				 *		<LI>Next you have the option of drawing the primary image. The primary image is normally the background image of the control. If you choose to draw this (either manually or\n
				 *			with a call to drawPrimaryImage you also have the option of using the default drawing routine (usually this involves drawing a bounding rect around the controls area\n
				 *			Default drawing option is set via the m_doDefaultDrawing protected variable.
				 *		<LI>Finally you would draw your controls specific graphics needs
				 * </OL>
				 * \n
				 * @code
				 * void CMyControl::drawControl(CGraphics &graphics)
				 * {
				 *		// First check if we can allow the standard handler to draw the disabled control
				 *		if (!this->drawEnabledControl(graphics))
				 *		{
				 *			return;
				 *		}
				 *		
				 *		// Draw the background
				 *		this->drawPrimaryImage(graphics, m_doDefaultDrawing);
				 *		
				 *		// At this point, do your specific drawing
				 * }
				 * @endcode
				 */
				virtual void drawControl(CGraphics &graphics);

//	===========================================================================

				/**
				 * Enable a control
				 */
				virtual void enableControl() ;

				/**
				 * Disable the control
				 */
				virtual void disableControl();

//	===========================================================================

				/**
				 * Set the primary (usually background) image
				 * @param theImage The image to draw as the primary image (usually the control background, but control specific)
				 */
				virtual void setPrimaryImage(IImage *theImage);

				/**
				 * Set the disabled image
				 * @param theImage The image to draw when the control is disabled
				 */
				virtual void setDisabledImage(IImage *theImage);

//	===========================================================================

				/**
				 * Handle a file drop
				 * @param event The event to handle
				 */
				virtual void handleFileDrop(const CDropEvent &event) { }

//	===========================================================================

				/**
				 * Handle key down events
				 * @param event The event to handle
				 */
				virtual bool handleKeyDown(const CKeyboardEvent &event) { return false; }
				
				/**
				 * Handle key up events
				 * @param event The event to handle
				 */
				virtual bool handleKeyUp(const CKeyboardEvent &event) { return false; }

//	===========================================================================

				/**
				 * Handle left button being clicked
				 * @param event The event to handle
				 */
				virtual void handleLeftButtonDown(CMouseEvent &event) { }

				/**
				 * Handle left button being released
				 * @param event The event to handle
				 */
				virtual void handleLeftButtonUp(CMouseEvent &event) { }

				/**
				 * Handle the right button being clicked
				 * @param event The event to handle
				 */
				virtual void handleRightButtonDown(CMouseEvent &event) { }

				/**
				 * Handle the right button being released
				 * @param event The event to handle
				 */
				virtual void handleRightButtonUp(CMouseEvent &event) { }

				/**
				 * Handle a double click on the left button
				 * @param event The event to handle
				 */
				virtual void handleDoubleClick(CMouseEvent &event) { }

				/**
				 * Handle the scroll wheel
				 * @param event The event to handle
				 */
				virtual void handleMouseScroll(CMouseEvent &event) { }

				/**
				 * Handle the mouse movement
				 * @param event The event to handle
				 */
				virtual void handleMouseMovement(CMouseEvent &event) { }

				/**
				 * Handle a mouse leaving the root area
				 * @param event The event to handle
				 */
				virtual void handleMouseLeavingArea(CMouseEvent &event) { }

//	===========================================================================

				/**
				 * Set the colours to draw with by default
				 * @param frameColour The colour of the frame
				 * @param backgroundColour The background colour of the control
				 */
				virtual void setDefaultControlColours(const CAlphaColour &backgroundColour = CAlphaColour::CALPHACOLOUR_WHITE, const CAlphaColour &frameColour = CAlphaColour::CALPHACOLOUR_BLACK);

				/**
				 * Should the default drawing take place?
				 * @param doDefaultDraw If true default drawing takes place, if false, drawing doesnt take place
				 */
				virtual void doDefaultDrawing(const bool doDefaultDraw) { m_doDefaultDrawing = doDefaultDraw; }

//	===========================================================================

				/**
				 * Increment the reference count
				 */
				virtual void referenced();

				/**
				 * Decrement the reference count. Object is deleted if reference count is <= 0
				 */
				virtual void dereference();

				/**
				 * Get the reference count
				 * @retval long The reference count
				 */
				virtual long getReferenceCount() const { return m_referenceCount; }

				/**
				 * Get a description of the object
				 * @param string On return is filled with the description
				 * @param size The size of the stirng
				 */
				virtual void getObjectDescription(char *string, const long size) const;

//	===========================================================================

			protected:

//	===========================================================================

				/**
				 * Check if the control is enabled. If it isnt, draw the disabled background and return
				 * @param graphics The graphics context to draw with
				 * @retval bool If true the control should be drawn, if false the control should not be drawn
				 */
				virtual bool drawEnabledControl(CGraphics &graphics);

				/**
				 * Draw the background
				 * @param graphics The graphics context to draw with
				 * @retval bool If true the control should be drawn, if false the control should not be drawn
				 */
				virtual bool drawPrimaryImage(CGraphics &graphics);

				/**
				 * Draw the background
				 * @param graphics The graphics context to draw with
				 * @param drawFrameIfNoImage If true the frame is drawn if there is no primary image
				 * @retval bool If true the control should be drawn, if false the control should not be drawn
				 */
				virtual void drawPrimaryImage(CGraphics &graphics, const bool drawFrameIfNoImage);

//	===========================================================================


				long m_referenceCount;					/**< How many things point to this item? */
				IActionListener *m_actionListener;		/**< The action listener for the control */
				IControlRoot *m_rootControl;			/**< The root control */
				IControl *m_parentControl;				/**< The parent container */

				bool m_mouseEnabled;					/**< Is the control able to accept mouse commands */
				bool m_keyboardEnabled;					/**< Is the control keyboard enabled */
				bool m_dropEnabled;						/**< Is the control drop enabled */
				bool m_needsRightClick;					/**< Does the control need a right click?? */
				bool m_enabled;							/**< is the control enabled?? */
				bool m_doDefaultDrawing;				/**< Should the default drawing take place? */

				long m_uniqueId;						/**< The identifier */
				double m_value;							/**< The current value (0 - 1.0 value) */

				CRect m_area;							/**< The area of the control, relative to the control roots top left (0, 0 point) */
				CDimension m_dimension;					/**< the dimension of the cotnrol */
				CRect m_absoluteArea;					/**< The absolute position of the control, relative to window's top left */
				CRect m_normalisedArea;					/**< The area as a 0, 0 position rect */

				IImage *m_primaryImage;					/**< The primary image for the control */
				IImage *m_disabledImage;				/**< The disabled image */

				CAlphaColour m_frameColour;				/**< The frame colour */
				CAlphaColour m_backgroundColour;		/**< The background colour */
			};
		}
	}
}
#endif	// End of CControl.hpp